/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.internal.deprecation;

import com.google.common.base.Joiner;
import org.gradle.api.problems.DocLink;
import org.gradle.api.problems.internal.InternalProblem;
import org.gradle.util.GradleVersion;
import org.gradle.util.internal.DefaultGradleVersion;
import org.jspecify.annotations.Nullable;

import javax.annotation.CheckReturnValue;
import java.util.List;

@SuppressWarnings("SameNameButDifferent")
@CheckReturnValue
public class DeprecationMessageBuilder<T extends DeprecationMessageBuilder<T>> {

    private static final GradleVersion GRADLE10 = GradleVersion.version("10.0.0");
    private static final GradleVersion GRADLE11 = GradleVersion.version("11.0.0");

    @Nullable
    protected String summary;
    private DeprecationTimeline deprecationTimeline;
    private String context;
    private String advice;
    private DocLink documentation = null;
    private DeprecatedFeatureUsage.Type usageType = DeprecatedFeatureUsage.Type.USER_CODE_DIRECT;

    protected String problemIdDisplayName;
    protected String problemId;

    public static WithDocumentation withDocumentation(InternalProblem warning, WithDeprecationTimeline withDeprecationTimeline) {
        DocLink docLink = warning.getDefinition().getDocumentationLink();
        if (docLink != null) {
            return withDeprecationTimeline
                .withDocumentation(warning.getDefinition().getDocumentationLink());
        }
        return withDeprecationTimeline.undocumented();
    }

    @Nullable
    protected String createDefaultDeprecationIdDisplayName() {
        return summary;
    }

    @SuppressWarnings("unchecked")
    public T withContext(String context) {
        this.context = context;
        return (T) this;
    }

    @SuppressWarnings("unchecked")
    public T withAdvice(String advice) {
        this.advice = advice;
        return (T) this;
    }

    @SuppressWarnings("unchecked")
    public T withProblemIdDisplayName(String problemIdDisplayName) {
        this.problemIdDisplayName = problemIdDisplayName;
        return (T) this;
    }

    @SuppressWarnings("unchecked")
    public T withProblemId(String problemId) {
        this.problemId = problemId;
        return (T) this;
    }

    /**
     * Output: This is scheduled to be removed in Gradle 10.
     */
    public WithDeprecationTimeline willBeRemovedInGradle10() {
        this.deprecationTimeline = DeprecationTimeline.willBeRemovedInVersion(GRADLE10);
        return new WithDeprecationTimeline(this);
    }

    /**
     * Output: This will fail with an error in Gradle 10.
     */
    public WithDeprecationTimeline willBecomeAnErrorInGradle10() {
        this.deprecationTimeline = DeprecationTimeline.willBecomeAnErrorInVersion(GRADLE10);
        return new WithDeprecationTimeline(this);
    }

    /**
     * Output: This will fail with an error in Gradle X.
     * <p>
     * Where X is the current major Gradle version + 1.
     *
     * NOTE: This should be used sparingly. It is better to use the version-specific methods for deprecations that will become errors.
     * This is intended for persistent deprecations that will never be removed.
     * As an example, Gradle will always have a deprecation about using a version of Java older than the future minimum version.
     */
    public WithDeprecationTimeline willBecomeAnErrorInNextMajorGradleVersion() {
        GradleVersion nextMajor = DefaultGradleVersion.current().getNextMajorVersion();
        this.deprecationTimeline = DeprecationTimeline.willBecomeAnErrorInVersion(nextMajor);
        return new WithDeprecationTimeline(this);
    }

    /**
     * Output: Starting with Gradle 10, ${message}.
     */
    public WithDeprecationTimeline startingWithGradle10(String message) {
        this.deprecationTimeline = DeprecationTimeline.startingWithVersion(GRADLE10, message);
        return new WithDeprecationTimeline(this);
    }

    /**
     * Output: Starting with Gradle 11, ${message}.
     */
    public WithDeprecationTimeline startingWithGradle11(String message) {
        this.deprecationTimeline = DeprecationTimeline.startingWithVersion(GRADLE11, message);
        return new WithDeprecationTimeline(this);
    }

    void setIndirectUsage() {
        this.usageType = DeprecatedFeatureUsage.Type.USER_CODE_INDIRECT;
    }

    void setBuildInvocationUsage() {
        this.usageType = DeprecatedFeatureUsage.Type.BUILD_INVOCATION;
    }

    void setSummary(@Nullable String summary) {
        this.summary = summary;
    }

    void setAdvice(String advice) {
        this.advice = advice;
    }

    void setDocumentation(DocLink documentation) {
        this.documentation = documentation;
    }

    void setProblemIdDisplayName(@Nullable String problemIdDisplayName) {
        this.problemIdDisplayName = problemIdDisplayName;
    }

    void setDeprecationTimeline(DeprecationTimeline deprecationTimeline) {
        this.deprecationTimeline = deprecationTimeline;
    }

    DeprecationMessage build() {
        if (problemIdDisplayName == null) {
            setProblemIdDisplayName(createDefaultDeprecationIdDisplayName());
        }

        return new DeprecationMessage(summary, deprecationTimeline.toString(), advice, context, documentation, usageType, problemIdDisplayName, problemId);
    }

    public void setProblemId(String problemId) {
        this.problemId = problemId;
    }

    public static class WithDeprecationTimeline extends Documentation.AbstractBuilder<WithDocumentation> {
        private final DeprecationMessageBuilder<?> builder;

        public WithDeprecationTimeline(DeprecationMessageBuilder<?> builder) {
            this.builder = builder;
        }

        @Override
        public WithDocumentation withDocumentation(DocLink documentation) {
            builder.setDocumentation(documentation);
            return new WithDocumentation(builder);
        }
    }

    public static class WithDocumentation {
        private final DeprecationMessageBuilder<?> builder;

        WithDocumentation(DeprecationMessageBuilder<?> builder) {
            this.builder = builder;
        }

        /**
         * Terminal operation. Emits the deprecation message.
         */
        public void nagUser() {
            DeprecationLogger.nagUserWith(builder, WithDocumentation.class);
        }
    }

    public static abstract class WithReplacement<T, SELF extends WithReplacement<T, SELF>> extends DeprecationMessageBuilder<SELF> {
        protected final String subject;
        private T replacement;

        WithReplacement(String subject) {
            this.subject = subject;
        }

        /**
         * Constructs advice message based on the context.
         *
         * deprecateProperty: Please use the ${replacement} property instead.
         * deprecateMethod/deprecateInvocation: Please use the ${replacement} method instead.
         * deprecatePlugin: Please use the ${replacement} plugin instead.
         * deprecateTask: Please use the ${replacement} task instead.
         * deprecateInternalApi: Please use ${replacement} instead.
         * deprecateNamedParameter: Please use the ${replacement} named parameter instead.
         */
        @SuppressWarnings("unchecked")
        public SELF replaceWith(T replacement) {
            this.replacement = replacement;
            return (SELF) this;
        }

        String formatSubject() {
            return subject;
        }

        abstract String formatSummary(String subject);

        abstract String formatAdvice(T replacement);


        @Override
        DeprecationMessage build() {
            setSummary(formatSummary(formatSubject()));
            if (replacement != null) {
                setAdvice(formatAdvice(replacement));
            }

            if (problemIdDisplayName == null) {
                setProblemIdDisplayName(summary);
            }
            if (problemId == null) {
                setProblemId(DeprecationMessageBuilder.createDefaultDeprecationId(createDefaultDeprecationIdDisplayName()));
            }

            return super.build();
        }
    }

    public static class DeprecateAction extends WithReplacement<String, DeprecateAction> {
        DeprecateAction(String subject) {
            super(subject);
        }

        @Override
        protected String createDefaultDeprecationIdDisplayName() {
            return subject;
        }

        @Override
        String formatSummary(String subject) {
            return String.format("%s has been deprecated.", subject);
        }

        @Override
        String formatAdvice(String replacement) {
            return String.format("Please use %s instead.", replacement);
        }
    }

    public static class DeprecateNamedParameter extends WithReplacement<String, DeprecateNamedParameter> {

        DeprecateNamedParameter(String parameter) {
            super(parameter);
        }

        @Override
        String formatSummary(String parameter) {
            return String.format("The %s named parameter has been deprecated.", parameter);
        }

        @Override
        String formatAdvice(String replacement) {
            return String.format("Please use the %s named parameter instead.", replacement);
        }
    }

    public static class DeprecateProperty extends WithReplacement<String, DeprecateProperty> {
        private final Class<?> propertyClass;
        private final String property;

        DeprecateProperty(Class<?> propertyClass, String property) {
            super(property);
            this.propertyClass = propertyClass;
            this.property = property;
        }

        /**
         * DO NOT CALL THIS METHOD
         */
        @Deprecated
        public WithDeprecationTimeline willBeRemovedInGradle9() {
            setDeprecationTimeline(DeprecationTimeline.willBeRemovedInVersion(GRADLE10));
            return new WithDeprecationTimeline(this);
        }

        @Override
        public WithDeprecationTimeline willBeRemovedInGradle10() {
            setDeprecationTimeline(DeprecationTimeline.willBeRemovedInVersion(GRADLE10));
            return new WithDeprecationTimeline(this);
        }

        public class WithDeprecationTimeline extends DeprecationMessageBuilder.WithDeprecationTimeline {
            private final DeprecateProperty builder;

            public WithDeprecationTimeline(DeprecateProperty builder) {
                super(builder);
                this.builder = builder;
            }

            /**
             * Output: See DSL_REFERENCE_URL for more details.
             */
            @CheckReturnValue
            public WithDocumentation withDslReference() {
                setDocumentation(Documentation.dslReference(propertyClass, property));
                return new WithDocumentation(builder);
            }
        }

        @Override
        String formatSubject() {
            return String.format("%s.%s", propertyClass.getSimpleName(), property);
        }

        @Override
        String formatSummary(String property) {
            return String.format("The %s property has been deprecated.", property);
        }

        @Override
        String formatAdvice(String replacement) {
            return String.format("Please use the %s property instead.", replacement);
        }
    }

    public static class DeprecateSystemProperty extends WithReplacement<String, DeprecateSystemProperty> {
        private final String systemProperty;

        DeprecateSystemProperty(String systemProperty) {
            super(systemProperty);
            this.systemProperty = systemProperty;
            // This never happens in user code
            setIndirectUsage();
        }

        @Override
        String formatSubject() {
            return systemProperty;
        }

        @Override
        String formatSummary(String property) {
            return String.format("The %s system property has been deprecated.", property);
        }

        @Override
        String formatAdvice(String replacement) {
            return String.format("Please use the %s system property instead.", replacement);
        }
    }

    @CheckReturnValue
    public static class ConfigurationDeprecationTypeSelector {
        private final String configuration;

        ConfigurationDeprecationTypeSelector(String configuration) {
            this.configuration = configuration;
        }

        public DeprecateConfiguration forArtifactDeclaration() {
            return new DeprecateConfiguration(configuration, ConfigurationDeprecationType.ARTIFACT_DECLARATION);
        }

        public DeprecateConfiguration forConsumption() {
            return new DeprecateConfiguration(configuration, ConfigurationDeprecationType.CONSUMPTION);
        }

        public DeprecateConfiguration forDependencyDeclaration() {
            return new DeprecateConfiguration(configuration, ConfigurationDeprecationType.DEPENDENCY_DECLARATION);
        }

        public DeprecateConfiguration forResolution() {
            return new DeprecateConfiguration(configuration, ConfigurationDeprecationType.RESOLUTION);
        }
    }

    public static class DeprecateConfiguration extends WithReplacement<List<String>, DeprecateConfiguration> {
        private final ConfigurationDeprecationType deprecationType;

        DeprecateConfiguration(String configuration, ConfigurationDeprecationType deprecationType) {
            super(configuration);
            this.deprecationType = deprecationType;
            if (!deprecationType.inUserCode) {
                setIndirectUsage();
            }
        }

        @Override
        String formatSummary(String configuration) {
            return String.format("The %s configuration has been deprecated for %s.", configuration, deprecationType.displayName());
        }

        @Override
        String formatAdvice(List<String> replacements) {
            if (replacements.isEmpty()) {
                return "Please " + deprecationType.usage + " another configuration instead.";
            }
            return String.format("Please %s the %s configuration instead.", deprecationType.usage, Joiner.on(" or ").join(replacements));
        }
    }

    public static final char DASH = '-';

    public static String createDefaultDeprecationId(String... ids) {
        StringBuilder sb = new StringBuilder();
        for (String id : ids) {
            if (id == null) {
                continue;
            }
            CharSequence cleanId = createDashedId(id);
            if (cleanId.length() > 0) {
                sb.append(cleanId);
                sb.append(DASH);
            }
        }
        removeTrailingDashes(sb);
        return sb.toString();
    }

    private static void removeTrailingDashes(StringBuilder sb) {
        while (sb.length() > 0 && sb.charAt(sb.length() - 1) == DASH) {
            sb.setLength(sb.length() - 1);
        }
    }

    private static CharSequence createDashedId(String id) {
        StringBuilder cleanId = new StringBuilder();
        boolean previousWasDash = false;
        for (int i = 0; i < id.length(); i++) {
            char c = id.charAt(i);
            if (Character.isLetter(c)) {
                previousWasDash = false;
                cleanId.append(Character.toLowerCase(c));
            } else {
                if (previousWasDash) {
                    continue;
                }
                cleanId.append(DASH);
                previousWasDash = true;
            }
        }
        return cleanId;
    }

    public static class DeprecateMethod extends WithReplacement<String, DeprecateMethod> {
        private final Class<?> methodClass;
        private final String methodWithParams;

        DeprecateMethod(Class<?> methodClass, String methodWithParams) {
            super(methodWithParams);
            this.methodClass = methodClass;
            this.methodWithParams = methodWithParams;
        }

        @Override
        String formatSubject() {
            return String.format("%s.%s", methodClass.getSimpleName(), methodWithParams);
        }

        @Override
        String formatSummary(String method) {
            return String.format("The %s method has been deprecated.", method);
        }

        @Override
        String formatAdvice(String replacement) {
            return pleaseUseThisMethodInstead(replacement);
        }

        private static String pleaseUseThisMethodInstead(String replacement) {
            return String.format("Please use the %s method instead.", replacement);
        }
    }

    public static class DeprecateInvocation extends WithReplacement<String, DeprecateInvocation> {

        DeprecateInvocation(String invocation) {
            super(invocation);
        }

        @Override
        String formatSummary(String invocation) {
            return String.format("Using method %s has been deprecated.", invocation);
        }

        @Override
        String formatAdvice(String replacement) {
            return DeprecateMethod.pleaseUseThisMethodInstead(replacement);
        }
    }

    public static class DeprecateType extends WithReplacement<String, DeprecateType> {

        DeprecateType(String type) {
            super(type);
        }

        @Override
        String formatSummary(String type) {
            return String.format("The %s type has been deprecated.", type);
        }

        @Override
        String formatAdvice(String replacement) {
            return String.format("Please use the %s type instead.", replacement);
        }
    }

    public static class DeprecateTask extends WithReplacement<String, DeprecateTask> {
        DeprecateTask(String task) {
            super(task);
        }

        @Override
        String formatSummary(String task) {
            return String.format("The %s task has been deprecated.", task);
        }

        @Override
        String formatAdvice(String replacement) {
            return String.format("Please use the %s task instead.", replacement);
        }
    }

    public static class DeprecateTaskType extends WithReplacement<Class<?>, DeprecateTaskType> {
        private final String path;

        DeprecateTaskType(String task, String path) {
            super(task);
            this.path = path;
        }

        @Override
        String formatSummary(String type) {
            return String.format("The task type %s (used by the %s task) has been deprecated.", type, path);
        }

        @Override
        String formatAdvice(Class<?> replacement) {
            return String.format("Please use the %s type instead.", replacement.getCanonicalName());
        }
    }

    public static class DeprecatePlugin extends WithReplacement<String, DeprecatePlugin> {

        private boolean externalReplacement = false;

        DeprecatePlugin(String plugin) {
            super(plugin);
        }

        @Override
        String formatSummary(String plugin) {
            return String.format("The %s plugin has been deprecated.", plugin);
        }

        @Override
        String formatAdvice(String replacement) {
            return externalReplacement ? String.format("Consider using the %s plugin instead.", replacement) : String.format("Please use the %s plugin instead.", replacement);
        }

        /**
         * Advice output: Consider using the ${replacement} plugin instead.
         */
        public DeprecatePlugin replaceWithExternalPlugin(String replacement) {
            this.externalReplacement = true;
            return replaceWith(replacement);
        }
    }

    public static class DeprecateInternalApi extends WithReplacement<String, DeprecateInternalApi> {
        DeprecateInternalApi(String api) {
            super(api);
        }

        @Override
        String formatSummary(String api) {
            return String.format("Internal API %s has been deprecated.", api);
        }

        @Override
        String formatAdvice(String replacement) {
            return String.format("Please use %s instead.", replacement);
        }
    }

    public static class DeprecateBehaviour extends DeprecationMessageBuilder<DeprecateBehaviour> {

        private final String behaviour;

        public DeprecateBehaviour(String behaviour) {
            this.behaviour = behaviour;
        }

        @Override
        DeprecationMessage build() {
            setSummary(String.format("%s This behavior has been deprecated.", behaviour));
            return super.build();
        }
    }
}
